import random
import matplotlib.pyplot as plt


learning = 'sample'     # The type of learning ('map' or 'sample')
bias = 0.6              # The preference for regular languages
variables = 2           # The number of different variables in the language
variants = 2            # The number of different variants each variable can take
noise = 0.05            # The probability of producing the wrong variant
population_size = 1000  # Size of population
teachers = 'single'     # Either 'single' or 'multiple' 
method = 'chain'        # Either 'chain' or 'replacement'



# Produces a variant for a particular language and random variable
def produce(language):
    variable = random.randrange(len(language))
    correct_variant = language[variable]
    if random.random() > noise:
        return [variable,correct_variant]
    else:
        possible_noise_variants = range(variants)
        possible_noise_variants.remove(correct_variant)
        noisy_variant = random.choice(possible_noise_variants)
        return [variable,noisy_variant]
        

# classifies a language as either regular (all variables expressed with the
# same variant) or irregular (multiple variants used)
def regular(language):
    regular = True
    first_variant = language[0]
    for variant in language:
        if variant != first_variant:
            regular = False
    return regular

# Gives the prior bias for a particular language. Note that this must sum to
# 1 for all languages, so there is some normalisation in here 
def prior(language):
    if regular(language):
        number_of_regular_languages = variants
        return bias / number_of_regular_languages 
    else:
        number_of_irregular_languages = pow(variants, variables) - variants
        return (1 - bias) / number_of_irregular_languages


# Calculates P(data | language)
def likelihood(data, language):
    total = 1.
    for utterance in data:
        variable = utterance[0]
        variant = utterance[1]
        if variant == language[variable]:
            total = total * (1. - noise)
        else:
            total = total * (noise / (variants - 1))
    return total


# Returns a list of all possible languages for expressing n variables
def all_languages(n):
    if n == 0:
        return [[]]
    else:
        result = []
        smaller_langs = all_languages(n - 1)
        for l in smaller_langs:
            for v in range(variants):
                result.append(l + [v])
        return result


# Picks a language give the posterior probabilities of all languages
# This will either be the maximum a posteriori language ('map')
# or a language sampled from the posterior
def select_language(data):    
    list_of_all_languages = all_languages(variables)
    list_of_posteriors = []
    for language in list_of_all_languages:
        this_language_posterior = likelihood(data,language) * prior(language) 
        list_of_posteriors.append(this_language_posterior)
    if learning == 'map':
        map_language_index = wta(list_of_posteriors)
        map_language = list_of_all_languages[map_language_index]
        return map_language
    if learning == 'sample':
        sampled_language_index = roulette_wheel(list_of_posteriors)
        sampled_language = list_of_all_languages[sampled_language_index]
        return sampled_language
        

# Good old winner-take-all
def wta(items):
    maxweight = max(items)
    candidates = []
    for i in range(len(items)):
        if items[i] == maxweight:
            candidates.append(i)
    return random.choice(candidates)

# Given a list of scores, returns a position in that list selected randomly
# in proportion to its score
def roulette_wheel(scores):
    summed_scores = sum(scores)
    r = random.uniform(0,summed_scores)
    accumulator = 0
    for i in range(len(scores)):
        accumulator += scores[i]
        if r < accumulator:
            return i
    
# Generates a new population, consisting of a specified number_of_learners,
# who learn from data generated by the adult population
def pop_learn(adult_population,bottleneck,number_of_learners):
    new_population = []
    for n in range(number_of_learners):
        if teachers == 'single':
            potential_teachers = [random.choice(adult_population)]
        if teachers == 'multiple':
            potential_teachers = adult_population
        data = []
        for n in range(bottleneck):
            teacher = random.choice(potential_teachers)
            utterance = produce(teacher)
            data.append(utterance)
        learner_grammar = select_language(data)
        new_population.append(learner_grammar)
    return new_population

# Returns a list of n randomly languages
def initial_population(n):
    population = []
    possible_languages = all_languages(variables)
    for agent in range(n):
        language=random.choice(possible_languages)
        population.append(language)
    return population

# Returns a list of two elements: final population, and accumulated
# data, which is expressed in temrs of proportion of the population using
# a regular language
def simulation(generations, bottleneck, report_every):
    population = initial_population(population_size)
    data_accumulator=[]
    for i in range(1,generations+1):
        if method == 'chain': # Replace whole population
            population = pop_learn(population, bottleneck, population_size)
        if method == 'replacement': #Replace one individual at a time
            population = population[1:] 
            new_agent = pop_learn(population, bottleneck, 1)[0]
            population.append(new_agent)
        if (i % report_every == 0):
            regular_language_count = 0
            for agent in population:
                if regular(agent):
                    regular_language_count += 1
            data_accumulator.append(regular_language_count / float(population_size))
    return [population,data_accumulator]

