"""
Simple learned signalling system simulation

learn takes a three arguments:
1. a signalling system (represented as a single matrix of
association weights, with meanings on rows, signals on columns),
2. a meaning (an integer),
3. a signal (also an integer)
and increases the association weight for that meaning-signal pair. 

train does the same but for a list of meaning-signal pairs.

Usage example:

system = [[0, 0, 0], [0, 0, 0], [0, 0, 0]]
train(system, [[0, 0], [1, 1], [2, 2]])

This example changes the matrix so that it has 1s on the diagonal.
ca_monte can be used as before to test whether an agent that has
learned a particular signalling system can talk to another one.
"""

import random
import matplotlib.pyplot as plt

def reception_weights(system, signal):
    weights = []
    for row in system:
        weights.append(row[signal])
    return weights

def production_weights(system, meaning):
    return system[meaning]

def wta(items):
    maxweight = max(items)
    candidates = []
    for i in range(len(items)):
        if items[i] == maxweight:
            candidates.append(i)
    return random.choice(candidates)


def communicate(speaker_system, hearer_system, meaning):
    speaker_signal = wta(production_weights(speaker_system,meaning))
    hearer_meaning = wta(reception_weights(hearer_system,speaker_signal))
    if meaning == hearer_meaning: 
        return 1
    else: 
        return 0


def ca_monte(speaker_system, hearer_system, trials):
    total = 0.
    accumulator = []
    for n in range(trials):
        total += communicate(speaker_system, hearer_system,
                             random.randrange(len(speaker_system)))
        accumulator.append(total/(n+1))
    return accumulator

# ----- new code for learning below -----

def learn(system, meaning, signal):
    system[meaning][signal] += 1

def train(system, word_list):
    for ms_pair in word_list:
        learn(system, ms_pair[0], ms_pair[1])

