"""
Cultural evolution by iterated learning

simulation creates an initial population of agents with all-zero connection
weights and uses this population to produce a data set of utterances.
These utterances are used to train the population at the next time step.
Different methods to update the population are included:

replacement - this implements the 'replacement method' whereby the oldest
agent is replaced by a new one each generation

chain - this implements a 'transmission chain' in which, at each time step the
whole population is replaced

closed - this implements the 'closed group' method, in which the population is
static and no new individuals are ever added

There are a number of global simulation parameters for simulation commented
below.

Usage example:

simulation(500, 1000, 10)

This runs the simulation for 500 'generations', with 1000 trials for the monte
carlo calculation, and returns a list of two elements: the final population,
and a list of 50 values, indicating communicative accuracy (as evaluated by
monte carlo simulations) for every tenth generation. This data can be plotted
using matplotlib.pyplot in the usual way.

""" 

import random
import matplotlib.pyplot as plt

def reception_weights(system, signal):
    weights = []
    for row in system:
        weights.append(row[signal])
    return weights

def production_weights(system, meaning):
    return system[meaning]

def wta(items):
    maxweight = max(items)
    candidates = []
    for i in range(len(items)):
        if items[i] == maxweight:
            candidates.append(i)
    return random.choice(candidates)


def communicate(speaker_system, hearer_system, meaning):
    speaker_signal = wta(production_weights(speaker_system,meaning))
    hearer_meaning = wta(reception_weights(hearer_system,speaker_signal))
    if meaning == hearer_meaning: 
        return 1
    else: 
        return 0

def learn(system, meaning, signal, rule):
    for m in range(len(system)):
        for s in range(len(system[m])):
            if m == meaning and s == signal: system[m][s] += rule[0]
            if m == meaning and s != signal: system[m][s] += rule[1]
            if m != meaning and s == signal: system[m][s] += rule[2]
            if m != meaning and s != signal: system[m][s] += rule[3]

def pop_learn(population, data, no_learning_episodes, rule):
    for n in range(no_learning_episodes):
        ms_pair = random.choice(data)
        learn(random.choice(population), ms_pair[0], ms_pair[1], rule)

def pop_produce(population, no_productions):
    ms_pairs = []
    for n in range(no_productions):
        speaker = random.choice(population)
        meaning = random.randrange(len(speaker))
        signal = wta(production_weights(speaker, meaning))
        ms_pairs.append([meaning,signal])
    return ms_pairs

def ca_monte_pop(population, trials):
    total = 0.
    for n in range(trials):
        speaker = random.choice(population)
        hearer = random.choice(population)
        total += communicate(speaker, hearer, random.randrange(len(speaker)))
    return total / trials

# ----- new code below -----

meanings = 5            # number of meanings
signals = 5             # number of signals
interactions = 100      # both the number of utterances produced and the number
                        # of times this set is randomly sampled for training.
size = 100              # size of population
method = 'replacement'  # method of population update
rule = [1, 0, 0, 0]     # learning rule (alpha, beta, gamma, delta)

def new_agent():
    system = []
    for row in range(meanings):
        row = []
        for column in range(signals):
            row.append(0)
        system.append(row)
    return system

def new_population(size):
    population = []
    for i in range(size):
        population.append(new_agent())
    return population

def simulation(generations, mc_trials, report_every):
    population = new_population(size)
    data_accumulator=[]
    for i in range(generations):
        data = pop_produce(population, interactions)
        if method == 'chain': 
            population = new_population(size)
            pop_learn(population, data, interactions, rule)
        if method == 'replacement':
            population = population[1:] #This removes the first item of the list
            learner=new_agent()
            pop_learn([learner], data, interactions, rule)
            population.append(learner)
        if method == 'closed':
            pop_learn(population, data, interactions, rule)
        if (i % report_every == 0):
            data_accumulator.append(ca_monte_pop(population, mc_trials))
    return [population,data_accumulator]
