"""
Cross-situational learning.

In this simulation, the learner doesn't have direct access to the speaker's
meaning - instead the context of use provides a set of candidate meanings.
Each learning episode may provide a different set of candidate meanings.

Over multiple learning episodes, the learner makes use of the cross-situational
information, by assuming that the 'true' meaning lies at the intersection of
these sets of candidate meanings.

There are a number of global simulation parameters commented below.

Usage example:

xsl_simulation(100, 1000, 10)

This runs the simulation for 500 episodes of learning, using 1000 trials for
the monte carlo calculation of communicative accuracy, and outputting data
every 50 learning episodes.
"""

import random
import matplotlib.pyplot as plt

def reception_weights(system, signal):
    weights = []
    for row in system:
        weights.append(row[signal])
    return weights

def production_weights(system, meaning):
    return system[meaning]

def wta(items):
    maxweight = max(items)
    candidates = []
    for i in range(len(items)):
        if items[i] == maxweight:
            candidates.append(i)
    return random.choice(candidates)


def communicate(speaker_system, hearer_system, meaning):
    speaker_signal = wta(production_weights(speaker_system,meaning))
    hearer_meaning = wta(reception_weights(hearer_system,speaker_signal))
    if meaning == hearer_meaning: 
        return 1
    else: 
        return 0

def ca_monte(speaker_system, hearer_system, trials):
    total = 0.
    for n in range(trials):
        total += communicate(speaker_system, hearer_system,
                             random.randrange(len(speaker_system)))
    return total / trials


# ------- new code below ------------

meanings = 5            # number of meanings
signals = 5             # number of signals
context_size = 3        # context size
rule = [1, 0, 0, 0]     # learning rule

# creates a new agent, with initial weights either all 0 (if type is given as
# "random"), or specifying an optimal system (if type is given as "optimal")
def new_agent(type):
    system = []
    for i in range(meanings):
        row = []
        for j in range(signals):
            if type == 'optimal':
                if i == j:
                    row.append(1)
                else:
                    row.append(0)
            if type == 'random':
                row.append(0)
        system.append(row)
    return system


# add random context to target meaning m
def add_context(m):
    m_list = range(meanings)
    if m in m_list:
        m_list.remove(m)
    random.shuffle(m_list) # randomizes list of non-target meanings
    context = m_list[ : (context_size-1)] # take first however many
    context.append(m) # add back in target meaning
    return context


# produce a single data items from speaker_system
def produce_data(speaker_system):
    meaning = random.randrange(len(speaker_system))
    signal = wta(production_weights(speaker_system,meaning))
    context = add_context(meaning)
    return [context,signal]


# learn a signal paired with multiple meanings
def multiple_meaning_learn(system,meaning_list,signal,rule):
    for m in range(len(system)):
        for s in range(len(system[m])):
            if m in meaning_list and s == signal: system[m][s] += rule[0]
            if m in meaning_list and s != signal: system[m][s] += rule[1]
            if m not in meaning_list and s == signal: system[m][s] += rule[2]
            if m not in meaning_list and s != signal: system[m][s] += rule[3]



def xsl_simulation(learning_episodes, trials, report_every):
    adult = new_agent('optimal')
    learner = new_agent('random')
    data_accumulator = []
    for i in range(learning_episodes):
        utterance = produce_data(adult)
        context = utterance[0]
        signal = utterance[1]
        multiple_meaning_learn(learner, context, signal, rule)
        if (i % report_every == 0):
            data_accumulator.append(ca_monte(adult, learner, trials))
    return [learner,data_accumulator]
